/* ScummVM Tools
 *
 * ScummVM Tools is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * Additionally this file is based on the ScummVM source code.
 * Copyright information for the ScummVM source code is
 * available in the COPYRIGHT file of the ScummVM source
 * distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef COMMON_ENDIAN_H
#define COMMON_ENDIAN_H

#include "common/scummsys.h"

#define MKTAG(a0,a1,a2,a3) ((uint32)((a3) | ((a2) << 8) | ((a1) << 16) | ((a0) << 24)))

#if defined(INVERSE_MKID)
#define MKID_BE(a) ((uint32) \
		(((a) >> 24) & 0x000000FF) | \
		(((a) >>  8) & 0x0000FF00) | \
		(((a) <<  8) & 0x00FF0000) | \
		(((a) << 24) & 0xFF000000))

#else
#  define MKID_BE(a) ((uint32)(a))
#endif

static inline uint64 SWAP_64(uint64 a) {
	return ((uint64)((((a) >> 56) & 0x000000FF) |
			 (((a) >> 40) & 0x0000FF00) |
			 (((a) >> 24) & 0x00FF0000) |
			 (((a) >>  8) & 0xFF000000) |
			 (((a) & 0xFF000000) <<  8) |
			 (((a) & 0x00FF0000) << 24) |
			 (((a) & 0x0000FF00) << 40) |
			 (((a) & 0x000000FF) << 56) ));
}

static inline uint32 SWAP_32(uint32 a) {
	return (uint32)
		(((a >> 24) & 0x000000FF) |
		 ((a >>  8) & 0x0000FF00) |
		 ((a <<  8) & 0x00FF0000) |
		 ((a << 24) & 0xFF000000));
}

static inline uint16 SWAP_16(uint16 a) {
	return uint16(((a >> 8) & 0xFF) | ((a << 8) & 0xFF00));
}

#if defined(SCUMM_BIG_ENDIAN)
	#define FROM_LE_64(a) SWAP_64(a)
	#define FROM_LE_32(a) SWAP_32(a)
	#define FROM_LE_16(a) SWAP_16(a)
	#define TO_LE_64(a) SWAP_64(a)
	#define TO_LE_32(a) SWAP_32(a)
	#define TO_LE_16(a) SWAP_16(a)
	#define FROM_BE_64(a) ((uint64)(a))
	#define FROM_BE_32(a) ((uint32)(a))
	#define FROM_BE_16(a) ((uint16)(a))
	#define TO_BE_64(a) ((uint64)(a))
	#define TO_BE_32(a) ((uint32)(a))
	#define TO_BE_16(a) ((uint16)(a))
#else
	#define FROM_LE_64(a) ((uint64)(a))
	#define FROM_LE_32(a) ((uint32)(a))
	#define FROM_LE_16(a) ((uint16)(a))
	#define TO_LE_64(a) ((uint64)(a))
	#define TO_LE_32(a) ((uint32)(a))
	#define TO_LE_16(a) ((uint16)(a))
	#define FROM_BE_64(a) SWAP_64(a)
	#define FROM_BE_32(a) SWAP_32(a)
	#define FROM_BE_16(a) SWAP_16(a)
	#define TO_BE_64(a) SWAP_64(a)
	#define TO_BE_32(a) SWAP_32(a)
	#define TO_BE_16(a) SWAP_16(a)
#endif

FORCEINLINE uint16 READ_LE_UINT16(const void *ptr) {
	const byte *b = (const byte *)ptr;
	return uint16((b[1] << 8) + b[0]);
}
FORCEINLINE uint32 READ_LE_UINT32(const void *ptr) {
	const byte *b = (const byte *)ptr;
	return (b[3] << 24) + (b[2] << 16) + (b[1] << 8) + (b[0]);
}
FORCEINLINE void WRITE_LE_UINT16(void *ptr, uint16 value) {
	byte *b = (byte *)ptr;
	b[0] = (byte)(value >> 0);
	b[1] = (byte)(value >> 8);
}
FORCEINLINE void WRITE_LE_UINT32(void *ptr, uint32 value) {
	byte *b = (byte *)ptr;
	b[0] = (byte)(value >>  0);
	b[1] = (byte)(value >>  8);
	b[2] = (byte)(value >> 16);
	b[3] = (byte)(value >> 24);
}

FORCEINLINE uint16 READ_BE_UINT16(const void *ptr) {
	const byte *b = (const byte *)ptr;
	return uint16((b[0] << 8) + b[1]);
}
FORCEINLINE uint32 READ_BE_UINT32(const void *ptr) {
	const byte *b = (const byte *)ptr;
	return uint32((b[0] << 24) + (b[1] << 16) + (b[2] << 8) + (b[3]));
}
FORCEINLINE void WRITE_BE_UINT16(void *ptr, uint16 value) {
	byte *b = (byte *)ptr;
	b[0] = (byte)(value >> 8);
	b[1] = (byte)(value >> 0);
}
FORCEINLINE void WRITE_BE_UINT32(void *ptr, uint32 value) {
	byte *b = (byte *)ptr;
	b[0] = (byte)(value >> 24);
	b[1] = (byte)(value >> 16);
	b[2] = (byte)(value >>  8);
	b[3] = (byte)(value >>  0);
}


#endif
